#ifndef lint
static const char rcsid[] = "$Id: zmodemt.c,v 1.1.1.1 2001/03/08 00:01:48 efalk Exp $" ;
#endif

/*
 * Copyright (c) 1995 by Edward A. Falk
 */


/**********
 *
 *
 *	@@@@@  @   @   @@@   @@@@   @@@@@  @   @  @@@@@  
 *	   @   @@ @@  @   @  @   @  @      @@ @@    @    
 *	  @    @ @ @  @   @  @   @  @@@    @ @ @    @    
 *	 @     @ @ @  @   @  @   @  @      @ @ @    @    
 *	@@@@@  @ @ @   @@@   @@@@   @@@@@  @ @ @    @    
 *
 *	ZMODEMT - transmit side of zmodem protocol
 *
 * transmit side of zmodem protocol
 *
 * Caller sets flags defined in zmodem.h as appropriate.
 * (default is basic zmodem)
 *
 * functions:
 *
 *	ZmTInit(ZModem *info)
 *	YmTInit(ZModem *info)
 *	XmTInit(ZModem *info)
 *		Initiate a connection
 *
 *	ZmTFile(char *filename, u_char flags[4], ZModem *info)
 *		Initiate a file transfer.  Flags are as specified
 *		under "ZCBIN" in zmodem.h
 *
 *	ZmTFinish(ZModem *info)
 *		last file
 *
 *	ZmTAbort(ZModem *info)
 *		abort transfer
 *
 * all functions return 0 on success, 1 on abort
 *
 *
 *	Edward A. Falk
 *
 *	January, 1995
 *
 *
 *
 **********/

#include "zmodem.h"
#include "crctab.h"

// static function definition ////////////////////////////////////////////////

static	int ZmSendFilename(ZModem *info);
static	int ZmGotRinit( ZModem *info );
static	int ZmSendZSInit( ZModem *info );
static	int ZmSendFileCrc( ZModem *info );
static	int ZmStartFileData(  ZModem *info );
static	int ZmSendFileData( ZModem *info );
static	int ZmGotSendAck( ZModem *info );
static	int ZmGotSendDoneAck( ZModem *info );
static	int ZmGotSendNak( ZModem *info );
static	int ZmSkipFile( ZModem *info );
static	int ZmGotSendPos( ZModem *info );
static	int ZmGotSendWaitAck( ZModem *info );
static	int ZmResendEof( ZModem *info );
static	int ZmOverAndOut( ZModem *info );
static	int YXmitData( u_char *buffer, int len, ZModem *info );
static	int YSendFilename( ZModem *info );
static	int YSendData( ZModem *info );
static	int YSendFin( ZModem *info );


static	u_char	zeros[4] = {0,0,0,0} ;


	/* called by user to establish protocol */

int ZmTInit( ZModem *info )
{
 int	err ;
 int	i ;

 info->state = TStart ;
 info->Protocol = ZMODEM;
 info->crc32 = 0 ;
 info->packetCount = 0 ;
 info->errCount = 0 ;
 info->escCtrl = info->escHibit = info->atSign = info->escape = 0 ;
 info->InputState = Idle;
 info->canCount = info->chrCount = 0 ;
 info->windowCount = 0 ;
 info->filename = NULL ;
 info->bufsize = 0 ;
 info->interrupt = 0 ;
 info->waitflag = 0 ;

 if( info->packetsize == 0 )
  {
   info->packetsize = 256 ;
  }

/* we won't be receiving much data, pick a reasonable buffer
 * size (largest packet will do)
 */

 i = info->packetsize * 2;
 if( i < 1024 ) i = 1024;
 if( i > sizeof(info->bufferalloc) )
  {
   i = sizeof(info->bufferalloc);
   info->packetsize = i / 2;
  }
 info->buffer = info->bufferalloc;

 ZIFlush(info) ;

	/* optional: send "rz\r" to remote end */
 if( DoInitRZ ) 
  {
   err = ZXmitStr((u_char *)"rz\r", 3, info);
   if(err) return(err);
  }

/* nudge receiver */
 err = ZXmitHdr(ZRQINIT, ZHEX, zeros, info);
 if(err) 
  {
   return err ;
  }

 info->timeout = 60 ;

#ifdef ZMODEMLOG
 zmodemlog("ZmTInit[%s]: sent ZRQINIT\n", ZmSname(info)) ;
#endif 								// ZMODEMLOG

 return(0);
}


/* called by user to establish Ymodem protocol */

int YmodemTInit( ZModem *info )
{
 info->state       = YTStart;
 info->Protocol    = YMODEM;
 info->errCount    = 0 ;
 info->InputState  = Ysend;
 info->canCount    = info->chrCount = 0 ;
 info->windowCount = 0 ;
 info->filename = NULL ;

 if( info->packetsize != 1024 )
  {
   info->packetsize = 128 ;
  }

 info->buffer = info->bufferalloc;

 ZIFlush(info) ;
 ZFlowControl(0, info) ;

 info->timeout = 60 ;

 return(0);
}


/* called by user to establish Xmodem protocol */

int XmodemTInit( ZModem *info )
{
 (void) YmodemTInit(info) ;
 info->Protocol = XMODEM;
 return 0 ;
}


/* called by user to begin transmission of a file */

int ZmTFile(char * file, char *rfile, u_int	f0, u_int f1, u_int f2,
	u_int f3, int	filesRem, int bytesRem,	ZModem	*info)
{
 if( file == NULL)
  {
   return(ZmErrCantOpen);
  }

 info->file = ZOpenFile(file, 0,info, 0);	/* open in read mode */
 if(info->file == NULL)
  {
   return(ZmErrCantOpen);
  }

 info->fileEof = 0 ;
 info->filename = file ;
 info->rfilename = (rfile != NULL) ? rfile : (char *) "noname" ;
 info->filesRem = filesRem ;
 info->bytesRem = bytesRem ;
 info->fileFlags[3] = f0 ;
 info->fileFlags[2] = f1 ;
 info->fileFlags[1] = f2 ;
 info->fileFlags[0] = f3 ;
 info->offset = info->lastOffset = 0 ;
 info->len = info->date = info->fileType = info->mode = 0 ;
 if( info->filename != NULL )
  {
    int len;
    int date;
    int mode;
    if(ZFileStat(info->filename, & len, & date, & mode)  == 0 ) {

     info->len  = len;
     info->date = date;
     info->fileType = 0 ;
     info->mode = (mode&0777)|0100000 ;
    }
  }

	if( info->Protocol == XMODEM )
	  return YSendData(info) ;

	if( info->Protocol == YMODEM )
	  return YSendFilename(info) ;

	info->state = FileWait ;
#ifdef ZMODEMLOG
	zmodemlog("ZmTFile[%s]: send ZFILE(%s)\n",
		ZmSname(info), info->rfilename) ;
#endif 								// ZMODEMLOG

	return(ZmSendFilename(info));
}

/* put long to buffer
 buf   = buffer
 pos   = current position
 size  = sizeof buffer
 val   = printed value
 radix = 8/10/16
*/    


static int ZmPutStr(u_char * buf, int * pos, int size, const char * str)
{
 if(buf == 0 || pos == 0 || str == 0)
  {
   return(0);
  }  

 while(1)
  {
   if(*pos + 1 >= size)
    {
     *(buf + (size - 1)) = '\0';
       return(1);
    } 

// 
   *(buf + *pos) = *str;
    if(*str++ == '\0')
     {
      return(1);
     } 
   *pos += 1;    
  }
}

static int ZmPutLong(u_char * buf, int * pos, int size, long val, int radix)
{
 u_char dig[24];			// temp buffer
 int num;				// position in temp buffer

// 
 if(
   buf == 0 || 				// invalid buffer pointer 
   pos == 0				// invalid buffer position
  )
  {
   return(0);
  }  

 if(val < 0)
  {
// 
   if(*pos < size)
    {
     *(buf + *pos) = '-';		// 
     *pos += 1;
    }  
// 
   val *= -1;
  }
 else
 if(val == 0)
  {
   if(*pos < size)
    {
     *(buf + *pos) = '0';
     *pos += 1;
    }  

   if(*pos < size)
    {
     *(buf + *pos) = '\0';
    }
   else
    {
     *(buf + (size - 1)) = '\0';
    }
//
   return(1);   
  }

// 
 num = 0;
 while(val != 0 && num < 24)
  {
   u_char ch;
   
   if(radix == 10)
    {
     ch = '0' + (val % 10);
     val /= 10;				// 
    }
   else
   if(radix ==  8)
    {
     ch = '0' + (val & 0x7);
     val >>= 3;				// 
    }    
   else
   if(radix == 16)
    {
     ch = (val & 0xF);
     if(ch <= 9)
      {	
       ch += '0';			//
      }
     else
      {
       ch = 'a' + (ch - 10);			//
      }
//
     val >>= 4;
    }

   dig[num++] = ch;
  }
    
 while(--num >= 0)
  {
   if(*pos < size)
    {
     *(buf + *pos) = dig[num];
    }
   else
    {
     *(buf + (size - 1)) = '\0';
      return(1);
    }
//
   *pos += 1;
  }

 *(buf + *pos) = '\0';
 return(1);
} 

/* send ZFILE header and filename & info.  Wait for response
 * from receiver.
 */


static	int ZmSendFilename( ZModem *info )
{
 int err;
 int pos;

 u_char	obuf[1024];

 info->state = FileWait ;

 err = ZXmitHdr(ZFILE, ZBIN, info->fileFlags, info);
 if(err)  { return(err); }

 pos = 0;
 while(pos < 1024)
  {
   obuf[pos] = *(info->rfilename + pos);
   if(obuf[pos] == '\0')
    {
     pos++; 
     break;
    } 
   pos++; 
  }
 
 if(pos >= 1024)
  {
   pos = 1023;
   obuf[pos] = '\0';
  } 


 ZmPutLong(obuf, & pos, 1024, info->len     , 10); 	/* %d */
 ZmPutStr (obuf, & pos, 1024, " "); 		

 ZmPutLong(obuf, & pos, 1024, info->date    ,  8); 	/* %lo */
 ZmPutStr (obuf, & pos, 1024, " "); 		

 ZmPutLong(obuf, & pos, 1024, info->mode    ,  8); 	/* %o */
 ZmPutStr (obuf, & pos, 1024, " "); 		

 ZmPutLong(obuf, & pos, 1024,  0            , 10); 	/*  0 */
 ZmPutStr (obuf, & pos, 1024, " "); 		

 ZmPutLong(obuf, & pos, 1024, info->filesRem, 10); 	/* %d */
 ZmPutStr (obuf, & pos, 1024, " "); 		

 ZmPutLong(obuf, & pos, 1024, info->bytesRem, 10); 	/* %d */
 ZmPutStr (obuf, & pos, 1024, " "); 		

 ZmPutLong(obuf, & pos, 1024,  0	    , 10); 	/*  0 */
 ZmPutStr (obuf, & pos, 1024, " "); 		

 if(pos >= 1024)
  {
   obuf[1023] = '\0';
   pos = 1023;
  }

 return(ZXmitData(ZBIN, pos + 1, ZCRCW, obuf, info));
}


/* called by user when there are no more files to send */

int ZmTFinish( ZModem *info )
{
	if( info->Protocol == XMODEM )
	  return ZmDone ;

	if( info->Protocol == YMODEM )
         {
	  return YSendFin(info) ;
         }

	info->state = TFinish ;
	if( info->buffer != NULL ) 
	 {
	  info->buffer = NULL ;
 	 }

#ifdef ZMODEMLOG
	zmodemlog("ZmTFinish[%s]: send ZFIN\n", ZmSname(info)) ;
#endif 								// ZMODEMLOG

	return ZXmitHdr(ZFIN, ZHEX, zeros, info) ;
}

//

		/* sent ZRQINIT, waiting for response */
	StateTable	TStartOps[] = {	
	  {ZRINIT,ZmGotRinit,1,1,TStart},
	  {ZCHALLENGE,ZmAnswerChallenge,1,0,TStart},
	  {ZABORT,ZmGotAbort,1,1,TFinish},
	  {ZFERR,ZmGotAbort,1,1,TFinish},
	  {ZNAK,ZmIgnore,0,0,TStart},
	  {ZCOMMAND,ZmGotCommand,0,0,CommandData},
	  {ZSTDERR,ZmGotStderr,0,0,StderrData},
	  {99,ZmZPF,0,0,TStart},
	} ;

		/* sent ZSINIT, waiting for response */
	StateTable	TInitOps[] = {
	  {ZACK,ZmRetDone,1,0,TInit},
	  {ZNAK,ZmSendZSInit,1,0,TInit},
	  {ZRINIT,ZmGotRinit,1,1,TInit},	/* redundant, but who cares */
	  {ZCHALLENGE,ZmAnswerChallenge,1,0,TInit},
	  {ZABORT,ZmGotAbort,1,1,TFinish},
	  {ZFERR,ZmGotAbort,1,1,TFinish},
	  {ZCOMMAND,ZmGotCommand,0,0,CommandData},
	  {ZSTDERR,ZmGotStderr,0,0,StderrData},
	  {99,ZmZPF,0,0,TInit},
	} ;

		/* sent ZFILE, waiting for response */
	StateTable	FileWaitOps[] = {
	  {ZRPOS,ZmSendFileData,1,0,Sending},
	  {ZSKIP,ZmSkipFile,1,0,FileWait},
	  {ZCRC,ZmSendFileCrc,1,0,FileWait},
	  {ZNAK,ZmSendFilename,1,0,FileWait},
	  {ZRINIT,ZmSendFilename,1,1,FileWait},	/* rcvr confused, retry file */
	  {ZABORT,ZmGotAbort,1,1,TFinish},
	  {ZFERR,ZmGotAbort,1,1,TFinish},
	  {ZCHALLENGE,ZmAnswerChallenge,1,0,FileWait},
	  {ZCOMMAND,ZmGotCommand,0,0,CommandData},
	  {ZSTDERR,ZmGotStderr,0,0,StderrData},
	  {99,ZmZPF,0,0,FileWait},
	} ;

		/* sent file CRC, waiting for response */
	StateTable	CrcWaitOps[] = {
	  {ZRPOS,ZmSendFileData,1,0,Sending},
	  {ZSKIP,ZmSkipFile,1,0,FileWait},
	  {ZNAK,ZmSendFileCrc,1,0,CrcWait},
	  {ZRINIT,ZmSendFilename,1,1,FileWait},	/* rcvr confused, retry file */
	  {ZABORT,ZmGotAbort,1,1,TFinish},
	  {ZFERR,ZmGotAbort,1,1,TFinish},
	  {ZCRC,ZmSendFileCrc,0,0,CrcWait},
	  {ZCHALLENGE,ZmAnswerChallenge,0,0,CrcWait},
	  {ZCOMMAND,ZmGotCommand,0,0,CommandData},
	  {ZSTDERR,ZmGotStderr,0,0,StderrData},
	  {99,ZmZPF,0,0,CrcWait},
	} ;

		/* sending data, interruptable */
	StateTable	SendingOps[] = {
	  {ZACK,ZmGotSendAck,0,0,Sending},
	  {ZRPOS,ZmGotSendPos,1,1,Sending},
	  {ZSKIP,ZmSkipFile,1,1,FileWait},
	  {ZNAK,ZmGotSendNak,1,1,Sending},
	  {ZRINIT,ZmSendFilename,1,1,FileWait},	/* rcvr confused, retry file */
	  {ZABORT,ZmGotAbort,1,1,TFinish},
	  {ZFERR,ZmGotAbort,1,1,TFinish},
	  {99,ZmZPF,0,0,SendWait},
	} ;

		/* sent data, need to send EOF */
	StateTable	SendDoneOps[] = {
	  {ZACK,ZmGotSendDoneAck,0,0,SendWait},
	  {ZRPOS,ZmGotSendPos,1,1,Sending},
	  {ZSKIP,ZmSkipFile,1,1,FileWait},
	  {ZNAK,ZmGotSendNak,1,1,Sending},
	  {ZRINIT,ZmSendFilename,1,1,FileWait},	/* rcvr confused, retry file */
	  {ZABORT,ZmGotAbort,1,1,TFinish},
	  {ZFERR,ZmGotAbort,1,1,TFinish},
	  {99,ZmZPF,0,0,SendWait},
	} ;

		/* sending data, waiting for ACK */
	StateTable	SendWaitOps[] = {
	  {ZACK,ZmGotSendWaitAck,0,0,Sending},
	  {ZRPOS,ZmGotSendPos,0,0,SendWait},
	  {ZSKIP,ZmSkipFile,1,1,FileWait},
	  {ZNAK,ZmGotSendNak,0,0,Sending},
	  {ZRINIT,ZmSendFilename,1,1,FileWait},	/* rcvr confused, retry file */
	  {ZABORT,ZmGotAbort,1,1,TFinish},
	  {ZFERR,ZmGotAbort,1,1,TFinish},
	  {99,ZmZPF,0,0,SendWait},
	} ;

		/* sent ZEOF, waiting for new RINIT */
	StateTable	SendEofOps[] = {
	  {ZRINIT,ZmSkipFile,1,0,TStart},	/* successful completion */
	  {ZACK,ZmIgnore,0,0,SendEof},	/* probably ACK from last packet */
	  {ZRPOS,ZmGotSendPos,1,1,SendWait},
	  {ZSKIP,ZmSkipFile,1,1,TStart},
	  {ZNAK,ZmResendEof,1,0,SendEof},
	  {ZRINIT,ZmSendFilename,1,1,FileWait},	/* rcvr confused, retry file */
	  {ZABORT,ZmGotAbort,1,1,TFinish},
	  {ZFERR,ZmGotAbort,1,1,TFinish},
	  {99,ZmZPF,0,0,SendEof},
	} ;

	StateTable	TFinishOps[] = {
	  {ZFIN,ZmOverAndOut,1,1,Done},
	  {ZNAK,ZmTFinish,1,1,TFinish},
	  {ZRINIT,ZmTFinish,1,1,TFinish},
	  {ZABORT,ZmGotAbort,1,1,TFinish},
	  {ZFERR,ZmGotAbort,1,1,TFinish},
	  {99,ZmZPF,0,0,TFinish},
	} ;


extern	char	*hdrnames[] ;



	/* Received Rinit.  Usually received while in start state,
	 * this can also be an attempt to resync after a protocol
	 * failure
	 */

static	int ZmGotRinit( ZModem *info )
{
	info->bufsize = info->hdrData[1] + info->hdrData[2]*256 ;
	info->zrinitflags = info->hdrData[4] + info->hdrData[3]*256 ;
	info->crc32 = info->zrinitflags & CANFC32 ;
	info->escCtrl = info->zrinitflags & ESCCTL ;
	info->escHibit = info->zrinitflags & ESC8 ;

	/* Full streaming: If receiver can overlap I/O, and if
	 * the sender can sample the reverse channel without hanging,
	 * and the receiver has not specified a buffer size, then we
	 * can simply blast away with ZCRCG packets.  If the receiver
	 * detects an error, it sends an attn sequence and a new ZRPOS
	 * header to restart the file where the error occurred.
	 *
	 * [note that zmodem8.doc does not define how much noise is
	 * required to trigger a ZCRCW packet.  We arbitrarily choose
	 * 64 bytes]
	 *
	 * If 'windowsize' is nonzero, and the receiver can do full
	 * duplex, ZCRCQ packets are sent instead of ZCRCG, to keep track
	 * of the number of characters in the queue.  If the queue fills
	 * up, we pause and wait for a ZACK.
	 *
	 *
	 * Full Streaming with Reverse Interrupt:  If sender cannot
	 * sample the input stream, then we define an Attn sequence
	 * that will be used to interrupt transmission.
	 *
	 *
	 * Full Streaming with Sliding Window:  If sender cannot
	 * sample input stream or respond to Attn signal, we send
	 * several ZCRCQ packets until we're sure the receiver must
	 * have sent back at least one ZACK.  Then we stop sending and
	 * read that ZACK.  Then we send one more packet and so on.
	 *
	 *
	 * Segmented Streaming:  If receiver cannot overlap I/O or can't do
	 * full duplex and has specified a maximum receive buffer size,
	 * whenever the buffer size is reached, we send a ZCRCW packet.
	 *
	 * TODO: what if receiver can't overlap, but hasn't set a buffer
	 * size?
	 *
	 * ZCRCE: CRC next, frame ends, header follows
	 * ZCRCG: CRC next, frame continues nonstop
	 * ZCRCQ: CRC next, send ZACK, frame continues nonstop
	 * ZCRCW: CRC next, send ZACK, frame ends, header follows
	 */

	ZFlowControl(1,info) ;

	if( (info->zrinitflags & (CANFDX|CANOVIO)) == (CANFDX|CANOVIO) &&
	    (SendSample||SendAttn)  &&
	    info->bufsize == 0 )
	{
	  if( info->windowsize == 0 )
           {
	    info->Streaming = Full;
           }
	  else
           {
	    info->Streaming = StrWindow;
           }
	}

	else if( (info->zrinitflags & (CANFDX|CANOVIO)) == (CANFDX|CANOVIO) &&
	    info->bufsize == 0 )
	{
	  info->Streaming = SlidingWindow;
	}

	else
	  info->Streaming = Segmented;

#ifdef ZMODEMLOG
	zmodemlog("GotRinit[%s]\n", ZmSname(info)) ;
#endif 								// ZMODEMLOG

	if( AlwaysSinit || info->zsinitflags != 0  ||  info->attn != NULL )
	  return(ZmSendZSInit(info));
	else
	  return ZmDone ;	/* caller now calls ZmTFile() */
}


static	int ZmSendZSInit( ZModem *info )
{
 int	err ;
 const char	*at = (info->attn != NULL) ? info->attn : "" ;
 u_char	fbuf[4] ;
 int pos;

 pos = 0;
 while( *(at + pos) != '\0' ) { pos += 1; } 

	/* TODO: zmodem8.doc states: "If the ZSINIT header specifies
	 * ESCCTL or ESC8, a HEX header is used, and the receiver
	 * activates the specified ESC modes before reading the following
	 * data subpacket." What does that mean?
	 */

#ifdef ZMODEMLOG
	zmodemlog("SendZSInit[%s]\n", ZmSname(info)) ;
#endif 								// ZMODEMLOG

	info->state = TInit ;
	fbuf[0] = fbuf[1] = fbuf[2] = 0 ;
	fbuf[3] = info->zsinitflags ;
        err = ZXmitHdr(ZSINIT, ZBIN, fbuf, info);
	if(err) { return(err); }
	err = ZXmitData(ZBIN, pos+1, ZCRCW, (u_char *)at, info);
	if(err) { return(err); } 

	return(0);
}


static	int ZmSendFileCrc( ZModem *info )
{
	u_long	crc ;

	crc = FileCrc(info->filename) ;

#ifdef ZMODEMLOG
	zmodemlog("SendFileCrc[%s]: %lx\n", ZmSname(info), crc) ;
#endif								//ZMODEMLOG
	
	return ZXmitHdrHex(ZCRC, ZEnc4(crc), info) ;
}


/* Utility: start sending data. */

static	int ZmStartFileData(  ZModem *info )
{
	int	err ;

	info->offset =
	info->lastOffset =
	info->zrposOffset = ZDec4(info->hdrData+1) ;

        ZFileSeek(info->file, info->offset, info);

	/* TODO: what if fseek fails?  Send EOF? */

#ifdef ZMODEMLOG
	zmodemlog("startFileData[%s]: %ld\n", ZmSname(info), info->offset) ;
#endif 								// ZMODEMLOG

        err = ZXmitHdr(ZDATA, ZBIN, info->hdrData+1, info);
	if(err) { return(err); }
	return (ZmSendMoreFileData(info));
}



	/* send a chunk of file data in response to a ZRPOS.  Whether this
	 * is followed by a ZCRCE, ZCRCG, ZCRCQ or ZCRCW depends on all
	 * sorts of protocol flags
	 */


static	int ZmSendFileData( ZModem *info )
{
	info->waitflag = 0 ;
	return(ZmStartFileData(info));
}






/* here if an ACK arrived while transmitting data.  Update
 * last known receiver offset, and try to send some more
 * data.
 */


static	int ZmGotSendAck( ZModem *info )
{
	u_long offset ;

	offset = ZDec4(info->hdrData+1) ;

	if( offset > info->lastOffset )
	  info->lastOffset = offset ;

#ifdef ZMODEMLOG
	zmodemlog("GotSendAck[%s]: %ld\n", ZmSname(info), info->offset) ;
#endif 								// ZMODEMLOG

	return 0 ;		/* DONT send more data, that will happen
				 * later anyway */
}



/* here if an ACK arrived after last file packet sent.  Send
 * the EOF.
 */


static	int ZmGotSendDoneAck( ZModem *info )
{
	u_long offset ;

	offset = ZDec4(info->hdrData+1) ;

	if( offset > info->lastOffset )
	  info->lastOffset = offset ;

#ifdef ZMODEMLOG
	zmodemlog("GotSendDoneAck[%s]: %ld\n", ZmSname(info), info->offset) ;
#endif							// 
	info->state = SendEof ;
	info->timeout = 60 ;
	return ZXmitHdrHex(ZEOF, ZEnc4(info->offset), info) ;
}


	/* off to a bad start; ZDATA header was corrupt.  Start
	 * from beginning
	 */


static	int ZmGotSendNak( ZModem *info )
{
	info->offset = info->zrposOffset ;

        ZFileSeek(info->file, info->offset, info);

	/* TODO: what if fseek fails?  Send EOF? */

#ifdef ZMODEMLOG
	zmodemlog("GotSendNak[%s]: %ld\n", ZmSname(info), info->offset) ;
#endif

	return (ZmSendMoreFileData(info));
}


	/* got a ZSKIP, receiver doesn't want this file.  */

static	int ZmSkipFile( ZModem *info )
{
#ifdef ZMODEMLOG
 zmodemlog("SkipFile[%s]\n", ZmSname(info)) ;
#endif 								// ZMODEMLOG

 ZCloseFile(info->file, info);
 return ZmDone ;
}


	/* got a ZRPOS packet in the middle of sending a file,
	 * set new offset and try again
	 */

static	int ZmGotSendPos( ZModem *info )
{
	ZStatus(DataErr, ++info->errCount, NULL) ;
	info->waitflag = 1 ;		/* next pkt should wait, to resync */

#ifdef ZMODEMLOG
	zmodemlog("GotSendPos[%s]\n", ZmSname(info), info->offset) ;
#endif 							// ZMODEMLOG
	return(ZmStartFileData(info));
}



	/* here if an ACK arrived while waiting while transmitting data.
	 * Update last known receiver offset, and try to send some more
	 * data.
	 */


static	int ZmGotSendWaitAck( ZModem *info )
{
	u_long	offset ;
	int	err ;

	offset = ZDec4(info->hdrData+1) ;

	if( offset > info->lastOffset )
	  info->lastOffset = offset ;

#ifdef ZMODEMLOG
	zmodemlog("GotSendWaitAck[%s]\n", ZmSname(info), offset) ;
#endif 								// ZMODEMLOG

	err = ZXmitHdr(ZDATA, ZBIN, ZEnc4(info->offset), info);
	if(err) return(err);

	return (ZmSendMoreFileData(info));
}



static	int ZmResendEof( ZModem *info )
{
	return ZXmitHdrHex(ZEOF, ZEnc4(info->offset), info) ;
}


static	int ZmOverAndOut( ZModem *info )
{
	ZXmitStr((u_char *)"OO", 2, info) ;
	return ZmDone ;
}





	/* YMODEM */


static	u_char	eotstr[1] = {EOT} ;


	/* ymodem parser */

int YsendChar( char c, ZModem *info )
{
	int	err ;

	if( info->canCount >= 2 ) {
	  ZStatus(RmtCancel, 0, NULL) ;
	  return ZmErrCancel ;
	}

	switch( info->state ) {
	  case YTStart:		/* wait for 'G', 'C' or NAK */
	    switch( c ) {
	      case 'G':		/* streaming YModem */
	      case 'C':		/* CRC YModem */
	      case NAK:		/* checksum YModem */
		info->PacketType = c ;
		return ZmDone ;
	      default:
		return 0 ;
	    }

	  case YTFile:		/* sent filename, waiting for ACK or NAK */
	    switch( c ) {
	      case NAK:		/* resend */
	      case 'C':
	      case 'G':
		ZStatus(DataErr, ++info->errCount, NULL) ;
		return YSendFilename(info) ;
	      case ACK:
		info->state = YTDataWait ;
	      default:
		return 0 ;
	    }

	  case YTDataWait:	/* sent filename, waiting for G,C or NAK */
	    switch( c ) {
	      case NAK:
	      case 'C':
	      case 'G':
		info->chrCount = 0 ;
		if( info->PacketType == 'G' )	/* send it all at once */
		{
		  while( info->state == YTData )
		   {
		    err = YSendData(info);	
		    if(err)
                     {
		      return(err);
                     }
                   } 
		  return 0 ;
		}
		else
		  return YSendData(info) ;
	      default:
		return 0 ;
	    }

	  case YTData:		/* sent data, waiting for ACK or NAK */
	    switch( c ) {
	      case 'C':
	      case 'G':		/* protocol failure, resend filename */
		if( info->Protocol == YMODEM ) {
		  ZStatus(DataErr, ++info->errCount, NULL) ;
		  info->state = YTFile ;
                  ZFileSeek(info->file, 0, info);
		  return YSendFilename(info) ;
		}
		/* else XModem, treat it like a NAK */
	      case NAK:
		ZStatus(DataErr, ++info->errCount, NULL) ;
		return YXmitData(info->buffer + info->bufp, info->ylen, info) ;
	      case ACK:
		info->offset += info->ylen ;
		info->bufp += info->ylen ;
		info->chrCount -= info->ylen ;
		ZStatus(SndByteCount, info->offset, NULL) ;
		return YSendData(info) ;
	      default:
		return 0 ;
	    }

	  case YTEOF:		/* sent EOF, waiting for ACK or NAK */
	    switch( c ) {
	      case NAK:
		return ZXmitStr(eotstr, 1, info) ;
	      case ACK:
// 
                if(info->Protocol == YMODEM)
                 {
		  info->state = YTStart;
		 }
		else
		 {
		  info->state = Done;
		 } 
		return ZmDone ;
	      default:
		return 0 ;
	    }

	  case YTFin:		/* sent Fin, waiting for ACK or NAK */
	    switch( c ) {
	      case NAK:
		return YSendFin(info) ;
	      case ACK:
		return ZmDone ;
	      default:
		return 0 ;
	    }
	  default:
	    return 0 ;
	}
}


static	int YXmitData( u_char *buffer, int len, ZModem *info )
{
 u_char	hdr[3] ;
 u_char	trail[2] ;
 u_long	crc = 0 ;
 int	i, err ;

 hdr[0] = len == 1024 ? STX : SOH ;
 hdr[1] = info->packetCount ;
 hdr[2] = ~hdr[1] ;
 err = ZXmitStr(hdr, 3, info);
 if(err) { return(err); }
 err = ZXmitStr(buffer, len, info);
 if(err) { return err; }

 if( info->PacketType == NAK ) {	/* checksum */
   for(i=0; i<len; ++i)
    crc += buffer[i] ;
	  trail[0] = crc % 256 ;
	  return ZXmitStr(trail,1, info) ;
	}
	else {
	  for(i=0; i<len; ++i)
	    crc = updcrc(buffer[i], crc) ;
	  crc = updcrc(0,crc) ; crc = updcrc(0,crc) ;
	  trail[0] = crc / 256 ;
	  trail[1] = crc % 256 ;
	  return ZXmitStr(trail,2, info) ;
	}
}


static	int YSendFilename( ZModem *info )
{
 int	i,len ;
 u_char	obuf[1024] ;
 u_char	*ptr = obuf ;
 int pos;

 if(info->PacketType != 'G')
  {
   info->state = YTFile;
  }
 else
  {
   info->state = YTDataWait;
  }
  
 info->packetCount = 0 ;
 info->offset = 0 ;

/* copy rfilename to obuf */
 pos = 0;
 while(1)
  {
   *(ptr + pos) = *(info->rfilename +pos);  
   if(*(ptr + pos) == '\0')
    {
     break;
    } 

   if(pos >= 1024 - 64)
    {
     *(ptr + pos) = '\0';
       break;
    }
// 
   pos += 1;
  }

 
 if(pos < 1023)
  {
   pos += 1;			/* skip terminating zero */
  }

 ZmPutLong(obuf, & pos, 1024, info->len , 10); /* %d */
 ZmPutStr (obuf, & pos, 1024, " ");

 ZmPutLong(obuf, & pos, 1024, info->date,  8); /* %lo */
 ZmPutStr (obuf, & pos, 1024, " ");

 ZmPutLong(obuf, & pos, 1024, info->mode,  8); /* %o */
 ZmPutStr (obuf, & pos, 1024, " ");

 ZmPutLong(obuf, & pos, 1024,          0, 10); /* 0 */

 if(pos < 1023)
  {
   pos += 1;			/* skip terminating zero */
  }


 ptr += pos;
 *ptr++ = '\0' ;

/* pad out to 128 bytes or 1024 bytes */
 i = ptr - obuf;
 len = i > 128 ? 1024 : 128;
 for(; i < len; ++i)
   *ptr++ = '\0';

 return(YXmitData(obuf, len, info));
}


	/* send next buffer of data */


static	int YSendData( ZModem *info )
{
	int	len;

	/* are there characters still in the read buffer? */

	if( info->chrCount <= 0 ) {
	  info->bufp = 0 ;
	  info->chrCount = 
	  ZFileRead(info->buffer, info->packetsize, info->file, info);
	  info->fileEof = ZFileEof(info->file,info) ;
	}

	if( info->chrCount <= 0 ) 
	 {
          ZCloseFile(info->file,info);
	  info->state = YTEOF ;
	  return ZXmitStr(eotstr, 1, info) ;
	}

	/* pad out to 128 bytes if needed */
	if( info->chrCount < 128 ) 
	 {
          int pos;
	  len = 128 - info->chrCount;
          pos = 0; 
          while(pos < len)
	   {
	    *(info->buffer + info->bufp + info->chrCount + pos) = 0x1a;
	   }
	  info->chrCount = 128 ;
	 }

	info->ylen = info->chrCount >= 1024 ? 1024 : 128 ;
	++info->packetCount ;

	info->state = YTData ;

	return YXmitData(info->buffer + info->bufp, info->ylen, info) ;
}



static	int YSendFin( ZModem *info )
{
 u_char	obuf[128];
 int pos;

 info->state = YTFin;
 info->packetCount = 0;
 for(pos = 0; pos < 128; ++pos)
  { 
   obuf[pos] = 0;
  }

 return(YXmitData(obuf, 128, info));
}



	/* utility: send a chunk of file data.  Whether this is followed
	 * by a ZCRCE, ZCRCG, ZCRCQ or ZCRCW depends on all
	 * sorts of protocol flags, plus 'waitflag'.  Exact amount of file
	 * data transmitted is variable, as escape sequences may pad out
	 * the buffer.
	 */


int ZmSendMoreFileData( ZModem *info )
{
 int type;
 int qfull = 0;
 int err;
 int len;		/* max # chars to send this packet */
 long pending;		/* # of characters sent but not acknowledged */

/* ZCRCE: CRC next, frame ends, header follows
 * ZCRCG: CRC next, frame continues nonstop
 * ZCRCQ: CRC next, send ZACK, frame continues nonstop
 * ZCRCW: CRC next, send ZACK, frame ends, header follows
 */

 if( info->interrupt ) 
  {
  /* Bugger, receiver sent an interrupt.  Enter a wait state
   * and see what they want.  Next header *should* be ZRPOS.
   */

   info->state   = SendWait;
   info->timeout = 60;
   return(0);
  }

/* Find out how many bytes we can transfer in the next packet */

 len = info->packetsize;

 pending = info->offset - info->lastOffset;

 if( info->windowsize != 0 && info->windowsize - pending <= len )
  {
   len = info->windowsize - pending ;
   qfull = 1 ;
  }

 if( info->bufsize != 0  &&  info->bufsize - pending <= len ) 
  {
   len = info->bufsize - pending ;
   qfull = 1 ;
  }

 if( len == 0 ) 
  {	/* window still full, keep waiting */
   info->state   = SendWait;
   info->timeout = 60;
   return(0);
  }


/* OK, we can safely transmit 'len' bytes of data.  Start reading
 * file until buffer is full.
 */

 len -= 10;		/* Pre-deduct 10 bytes for trailing CRC */


/* find out what kind of packet to send */
 if( info->waitflag ) 
  {
   type = ZCRCW;
   info->waitflag = 0;
  }

#ifdef	COMMENT
 else if( info->fileEof )
  type = ZCRCE ;
#endif	/* COMMENT */
 else if( qfull )
   type = ZCRCW ;
 else
 switch( info->Streaming ) 
  {
   case Full:
   case Segmented: type = ZCRCG ; break ;

   case StrWindow:
    if( (info->windowCount += len) < info->windowsize/4 )
      type = ZCRCG ;
    else {
      type = ZCRCQ ;
      info->windowCount = 0 ;
     }
    break ;

   default:
    case SlidingWindow: type = ZCRCQ ; break ;
  }


 {
  int	crc32 = info->crc32 ;
  int c=0, c2, atSign=0 ;
  u_long crc ;
  u_char *ptr = info->buffer ;

  crc = crc32 ? 0xffffffff : 0 ;

	  /* read characters from file and put into buffer until buffer is
	   * full or file is exhausted
	   */

  while( len > 0 )	/*   && (c = getc(info->file)) != EOF */
  {
    u_char ch;
    ZFileRead(&ch, 1, info->file, info);
    if(ZFileEof(info->file, info))
     {
      break;
     }
    c = ch;  


    if( !crc32 )
      crc = updcrc(c, crc) ;
    else
      crc = UPDC32(c, crc) ;

    /* zmodem protocol requires that CAN(ZDLE), DLE, XON, XOFF and
     * a CR following '@' be escaped.  In addition, I escape '^]'
     * to protect telnet, "<CR>~." to protect rlogin, and ESC for good
     * measure.
     */

    c2 = c & 0177 ;
    if( c == ZDLE || c2 == 020 || c2 == 021 || c2 == 023 ||
	c2 == 0177  ||  c2 == '\r'  ||  c2 == '\n'  ||  c2 == 033  ||
	c2 == 035  || (c2 < 040 && info->escCtrl) )
    {
      *ptr++ = ZDLE ;
      if( c == 0177 )
	*ptr = ZRUB0 ;
      else if( c == 0377 )
	*ptr = ZRUB1 ;
      else
	*ptr = c^0100 ;
      len -= 2 ;
    }
    else {
      *ptr = c ;
      --len ;
    }
    ++ptr ;

    atSign = c2 == '@' ;
    ++info->offset ;
  }

  /* if we've reached file end, a ZEOF header will follow.  If
   * there's room in the outgoing buffer for it, end the packet
   * with ZCRCE and append the ZEOF header.  If there isn't room,
   * we'll have to do a ZCRCW
   */

// 
  if(ZFileEof(info->file, info))
   {
    info->fileEof = 1;
   }
  else
   {
    info->fileEof = 0;
   }

// 
  if(info->fileEof) 
   {
    if( qfull  ||  (info->bufsize != 0 && len < 24) )
      type = ZCRCW ;
    else
      type = ZCRCE ;
   }

  *ptr++ = ZDLE ;
  if( !crc32 )
    crc = updcrc(type, crc) ;
  else
    crc = UPDC32(type, crc) ;
  *ptr++ = type ;

  if( !crc32 ) {
    crc = updcrc(0,crc) ; crc = updcrc(0,crc) ;
    ptr = putZdle(ptr, (crc>>8)&0xff, info) ;
    ptr = putZdle(ptr, crc&0xff, info) ;
  }
  else {
    crc = ~crc ;
    for(len=4; --len >= 0; crc >>= 8)
      ptr = putZdle(ptr, crc&0xff, info) ;
  }

  len = ptr - info->buffer ;
 }

 ZStatus(SndByteCount, info->offset, NULL) ;

 err = ZXmitStr(info->buffer, len, info);
 if(err)
  {
   return(err);
  }

#ifdef	COMMENT
 if( (err = ZXmitData(ZBIN, info->buffer, len, type, info)) )
   return err ;
#endif	/* COMMENT */

/* finally, do we want to wait after this packet? */

 switch( type ) 
  {
   case ZCRCE:
    info->state = SendEof ;
    info->timeout = 60 ;
    return ZXmitHdrHex(ZEOF, ZEnc4(info->offset), info) ;

   case ZCRCW:
    if(info->fileEof)
     {
      info->state = SendDone;
     }
    else
     {
      info->state = SendWait;
     } 
    info->timeout = 60 ;
    break ;

   default:
    info->state = Sending ;
    info->timeout = 0 ;
    break ;
  }


#ifdef	COMMENT
 if( info->fileEof ) {	/* Yes, file is done, send EOF and wait */
  info->state = SendEof ;
  info->timeout = 60 ;
  return ZXmitHdrHex(ZEOF, ZEnc4(info->offset), info) ;
 }
 else if( type == ZCRCW ) {
  info->state = SendWait ;
  info->timeout = 60 ;
 }
 else {
  info->state = Sending ;
  info->timeout = 0 ;
 }
#endif	/* COMMENT */
 return(0);
}
